import os
import re
import unicodedata
import tkinter as tk
from tkinter import ttk, filedialog, messagebox


# ---------- Safety helpers ----------
def tk_safe(s: str) -> str:
    """Make a string safe to render in Tk widgets."""
    if not isinstance(s, str):
        try:
            s = str(s)
        except Exception:
            return "�"
    s = "".join(ch if ord(ch) <= 0xFFFF else "�" for ch in s)
    s = re.sub(r"[\x00-\x1F\x7F]", "", s)
    return s


def unique_path(dirpath: str, name: str) -> str:
    """Ensure the target path doesn't collide with an existing entry."""
    base, ext = os.path.splitext(name)
    candidate = name
    i = 1
    while os.path.exists(os.path.join(dirpath, candidate)):
        candidate = f"{base} ({i}){ext}"
        i += 1
    return os.path.join(dirpath, candidate)


# ---------- Filename sanitizer ----------
def sanitize_filename(name: str) -> str:
    """Clean up a filename by removing/replacing unsafe or weird characters (NTFS-safe & Linux-friendly)."""

    # Normalize and skip dotfiles (e.g. .bashrc, .config)
    name = unicodedata.normalize("NFKC", name)
    if name.startswith("."):
        return name

    # Drop ASCII control chars
    name = re.sub(r"[\x00-\x1F\x7F]", "", name)
 
    # Replace colons (ASCII/fullwidth) with hyphen
    name = re.sub(r"[:：]", "-", name)

    # Replace all slash-like characters (ASCII + common Unicode variants) with a plain hyphen
    name = re.sub(r"[\\/∕⁄⧸⧹／＼]", "-", name)

    
    # Replace colons (ASCII/fullwidth) and slashes (both directions) with hyphen
    name = re.sub(r"[:：/\\]", "-", name)

    # Replace other reserved characters
    name = re.sub(r'[<>"|?*]', "-", name)

    # Replace miscellaneous symbols that often cause trouble
    name = name.replace("#", "-")

    # Normalize quotes to single quote (more portable)
    name = (
        name.replace("“", "'")
            .replace("”", "'")
            .replace("‘", "'")
            .replace("’", "'")
            .replace("＂", "'")
            .replace('"', "'")
    )

    # Collapse whitespace
    name = re.sub(r"\s+", " ", name).strip()

    # Avoid dangerous leading dashes
    if name.startswith("-"):
        name = name.lstrip("-").strip()

    # Strip trailing spaces or dots (illegal on NTFS)
    name = name.rstrip(" .")

    # If everything got stripped, keep something
    if not name or set(name) == {"."}:
        name = "untitled"

    # Keep reasonable length
    if len(name) > 240:
        base, ext = os.path.splitext(name)
        name = (base[:240 - len(ext)] + ext).rstrip(" .")

    # ---------- Normalize multiple hyphens ----------
    name = re.sub(r'\s*-\s*-\s*', ' - ', name)
    name = re.sub(r'\s*-{2,}\s*', ' - ', name)

    # ---------- Clean punctuation spacing safely ----------
    base, ext = os.path.splitext(name)

    # Remove spaces before punctuation marks
    base = re.sub(r'\s+([,.;:!?])', r'\1', base)

    # Ensure a single space after punctuation (but only within base name)
    base = re.sub(r'([,;:!?])(?=[^\s])', r'\1 ', base)

    # Collapse multiple periods inside base, but not extension
    base = re.sub(r"\.{2,}", ".", base)

    # Recombine
    name = f"{base}{ext}"

    # ---------- Final cleanups ----------
    # Collapse multiple dots overall, preserve extension
    if "." in name:
        parts = name.split(".")
        if len(parts) > 2:
            base = ".".join(parts[:-1])
            ext = parts[-1]
            base = re.sub(r"\.{2,}", ".", base)
            name = f"{base}.{ext}"
        else:
            name = re.sub(r"\.{2,}", ".", name)
    else:
        name = re.sub(r"\.{2,}", ".", name)

    # Final whitespace cleanup
    name = re.sub(r"\s+", " ", name).strip()

    return name


# ---------- UI actions ----------
def browse_directory():
    folder = filedialog.askdirectory()
    if folder:
        directory_var.set(folder)
        preview_files()


def preview_files():
    folder = directory_var.get()
    if not folder:
        return

    for iid in file_list.get_children():
        file_list.delete(iid)

    needing_fix = 0
    try:
        entries = os.listdir(folder)
    except Exception as e:
        messagebox.showerror("Error", f"Cannot read directory:\n{folder}\n\n{e}")
        return

    for entry in entries:
        try:
            sanitized = sanitize_filename(entry)
            if entry != sanitized:
                file_list.insert("", "end", values=(tk_safe(entry),))
                needing_fix += 1
        except Exception as e:
            file_list.insert("", "end", values=(f"[UNREADABLE] {tk_safe(entry)}",))
            log_message(f"PREVIEW ERROR on {entry}: {e}", "error")

    if needing_fix == 0:
        file_list.insert("", "end", values=("All files OK",))


def log_message(msg, color=None):
    try:
        log_text.insert("end", tk_safe(msg) + "\n", color or ())
        log_text.see("end")
    except Exception:
        try:
            log_text.insert("end", "�\n")
            log_text.see("end")
        except Exception:
            pass


def apply_rename():
    folder = directory_var.get()
    if not folder:
        messagebox.showwarning("No folder", "Please select a folder first.")
        return

    renamed = 0
    errors = 0
    log_text.delete("1.0", "end")

    try:
        entries = os.listdir(folder)
    except Exception as e:
        messagebox.showerror("Error", f"Cannot read directory:\n{folder}\n\n{e}")
        return

    for entry in entries:
        old_path = os.path.join(folder, entry)
        try:
            new_name = sanitize_filename(entry)
            if entry == new_name:
                continue
            new_path = unique_path(folder, new_name)
            try:
                os.rename(old_path, new_path)
                renamed += 1
                log_message(f"Renamed: {entry}  →  {os.path.basename(new_path)}", "success")
            except Exception as e:
                errors += 1
                log_message(f"ERROR: {entry} ({e})", "error")
        except Exception as e:
            errors += 1
            log_message(f"SANITIZE ERROR: {entry} ({e})", "error")

    preview_files()
    summary = f"--- {renamed} Files Renamed. Errors: {errors} ---"
    status_var.set(summary)
    log_message(summary, "summary")


# ---------- UI setup ----------
root = tk.Tk(className="sanitizeapp")
root.title("Directory Filename Sanitizer")
root.geometry("900x700")
root.minsize(900, 600)

top_frame = ttk.Frame(root, padding=10)
top_frame.pack(fill="x")

ttk.Label(top_frame, text="Selected folder:").pack(side="left")
directory_var = tk.StringVar()
ttk.Entry(top_frame, textvariable=directory_var, width=70).pack(side="left", padx=5)
ttk.Button(top_frame, text="Browse", command=browse_directory).pack(side="left")

status_frame = ttk.Frame(root, padding=10)
status_frame.pack(side="bottom", fill="x")


def clear_all():
    directory_var.set("")
    for iid in file_list.get_children():
        file_list.delete(iid)
    log_text.delete("1.0", "end")
    status_var.set("Idle")


def copy_all_log():
    text = log_text.get("1.0", "end-1c")
    if not text.strip():
        messagebox.showinfo("Copied", "Log is empty.")
        return
    root.clipboard_clear()
    root.clipboard_append(text)
    messagebox.showinfo("Copied", "Entire log copied to clipboard.")


ttk.Button(status_frame, text="Preview", command=preview_files).pack(side="left", padx=5)
ttk.Button(status_frame, text="Apply Rename", command=apply_rename).pack(side="left", padx=5)

for text, cmd in [("Clear", clear_all), ("Copy All Log", copy_all_log)]:
    tk.Button(
        status_frame,
        text=text,
        bg="#2083FD",
        fg="white",
        activebackground="#3B94FF",
        activeforeground="white",
        relief="raised",
        command=cmd
    ).pack(side="left", padx=5)

status_var = tk.StringVar(value="Idle")
tk.Label(status_frame, textvariable=status_var, font=("TkDefaultFont", 12, "bold"), fg="blue").pack(side="right")

paned = ttk.PanedWindow(root, orient="vertical")
paned.pack(fill="both", expand=True, padx=10, pady=10)

preview_frame = ttk.Frame(paned)
file_list = ttk.Treeview(preview_frame, columns=("NeedsFix",), show="headings")
file_list.heading("NeedsFix", text="Filenames Requiring Sanitization")
file_list.column("NeedsFix", width=700)
scrollbar1 = ttk.Scrollbar(preview_frame, orient="vertical", command=file_list.yview)
file_list.configure(yscroll=scrollbar1.set)
file_list.pack(side="left", fill="both", expand=True)
scrollbar1.pack(side="right", fill="y")
paned.add(preview_frame, weight=1)

log_frame = ttk.LabelFrame(paned, text="Rename Log")
log_text = tk.Text(log_frame, wrap="none", bg="#f9f9f9", undo=False, autoseparators=False)
scrollbar2 = ttk.Scrollbar(log_frame, orient="vertical", command=log_text.yview)
log_text.configure(yscroll=scrollbar2.set)
log_text.pack(side="left", fill="both", expand=True)
scrollbar2.pack(side="right", fill="y")


def block_edit(event):
    if event.keysym in ("Up", "Down", "Left", "Right", "Shift_L", "Shift_R"):
        return None
    if event.state & 0x4 and event.keysym.lower() == "c":
        try:
            root.clipboard_clear()
            root.clipboard_append(log_text.selection_get())
        except tk.TclError:
            pass
        return "break"
    return "break"


log_text.bind("<Key>", block_edit)
log_text.bind("<Button-3>", lambda e: "break")

log_text.tag_configure("success", foreground="green")
log_text.tag_configure("error", foreground="red")
log_text.tag_configure("skip", foreground="orange")
log_text.tag_configure("summary", foreground="blue", font=("TkDefaultFont", 10, "bold"))

paned.add(log_frame, weight=1)

root.mainloop()
