 #!/usr/bin/env python3
import sys
import os
import tkinter as tk
from tkinter import ttk, filedialog, messagebox
import subprocess, threading, time, re

# -------------------------------
# Global sudo password storage
# -------------------------------
sudo_password = None

# -------------------------------
# Root helper plumbing
# -------------------------------
def run_root_task(args_list, timeout=30):
    """Run privileged command using cached sudo password."""
    global sudo_password
    if not sudo_password:
        return "⚠ No sudo password set."
    try:
        cmd = ["sudo", "-S"] + args_list
        proc = subprocess.run(
            cmd,
            input=sudo_password + "\n",
            capture_output=True,
            text=True,
            timeout=timeout
        )
        out = (proc.stdout or "") + (proc.stderr or "")
        out = out.strip()
        if proc.returncode != 0:
            raise subprocess.CalledProcessError(proc.returncode, cmd, output=out)
        return out or "(no output)"
    except subprocess.TimeoutExpired:
        return "Root task timed out."
    except subprocess.CalledProcessError as e:
        return e.output.strip() or f"Root task failed with code {e.returncode}"

# -------------------------------
# Utility helpers
# -------------------------------
def run_cmd(cmd):
    try:
        result = subprocess.run(
            cmd, shell=True, capture_output=True, text=True, timeout=8
        )
        return result.stdout.strip() or result.stderr.strip() or ""
    except subprocess.TimeoutExpired:
        return ""

def run_smartctl_bundle(dev):
    """Run smartctl once with -i -A -H and return all output (on base device)."""
    return run_root_task(["smartctl", "-i", "-A", "-H", dev])

# -----------------------------------------
# SMART text interpretation helpers
# -----------------------------------------
def interpret_smart_summary(smart_text):
    analysis = []
    reallocated_count = None
    temperature_val = None

    # --- Reallocated sector count ---
    for line in smart_text.splitlines():
        if "Reallocated_Sector_Ct" in line:
            try:
                reallocated_count = int(line.split()[-1])
                if reallocated_count == 0:
                    analysis.append("Media health is excellent – no bad sectors remapped.")
                else:
                    analysis.append("Drive has reallocated sectors – monitor closely, risk is rising.")
                    analysis.append(f"Reallocated sectors detected: {reallocated_count}")
            except:
                pass
            break

    # --- Power-on hours ---
    hours = None
    for line in smart_text.splitlines():
        if "Power_On_Hours" in line:
            try:
                hours = int(line.split()[-1])
            except:
                pass
    if hours is not None:
        if hours > 30000:
            analysis.append(f"Drive has {hours} hours – beyond typical design life. Higher failure risk.")
        elif hours > 20000:
            analysis.append(f"Drive has {hours} hours – mid-life, expect eventual wear.")
        else:
            analysis.append(f"Drive has {hours} hours – relatively young, low wear.")

    # --- Temperature extraction + interpretation (universal fix) ---
    for line in smart_text.splitlines():
        if re.search(r"Temperature", line, re.I):
            nums = re.findall(r"\d+", line)
            if not nums:
                continue
            # Match numeric right after the dash but before parentheses, e.g. "-       28 (Min/Max 20/72)"
            match = re.search(r"-\s+(\d+)\s*(?:\(|$)", line)
            if match:
                temperature_val = int(match.group(1))
            else:
                # Fallback: choose the last plausible temperature (≤80 °C)
                candidates = [int(n) for n in nums if 0 <= int(n) <= 80]
                if candidates:
                    temperature_val = candidates[-1]
            break

    if temperature_val is not None:
        if temperature_val <= 35:
            temp_comment = "Excellent – drive running cool."
        elif temperature_val <= 45:
            temp_comment = "Temperature is within normal range."
        elif temperature_val <= 55:
            temp_comment = "Slightly warm but acceptable."
        elif temperature_val <= 60:
            temp_comment = "High temperature – monitor closely."
        else:
            temp_comment = "Too high – potential thermal risk."
        analysis.append(f"Drive temperature: {temperature_val} °C – {temp_comment}")
    else:
        analysis.append("Temperature data not found in SMART output.")

    # --- SMART overall firmware health ---
    if "SMART overall-health self-assessment test result: PASSED" in smart_text or " PASSED" in smart_text:
        analysis.append("SMART firmware check: PASSED – no internal warnings.")
    elif "FAILED" in smart_text:
        analysis.append("SMART firmware check: FAILED – replace immediately.")

    return "\n".join(analysis) if analysis else "No interpretable SMART fields found."

def run_smart_selftest(dev, test_type="short"):
    return run_root_task(["smartctl", "-t", test_type, dev])

def fetch_selftest_log(dev):
    return run_root_task(["smartctl", "-l", "selftest", dev])

SELFTEST_STATUS_PATTERNS = [
    (re.compile(r"Completed without error", re.I), ("success", "✅ Test completed without error.")),
    (re.compile(r"Completed: read failure|read failure", re.I), ("fail", "❌ Test failed – read failure detected.")),
    (re.compile(r"Completed:.*?with.*?error", re.I), ("fail", "❌ Test completed with error.")),
    (re.compile(r"Aborted", re.I), ("caution", "⚠️ Test aborted (often due to activity).")),
    (re.compile(r"Interrupted|Cancelled", re.I), ("caution", "⚠️ Test interrupted/cancelled.")),
    (re.compile(r"In progress|Self-test routine in progress", re.I), ("info", "ℹ️ Test is still in progress.")),
]

def parse_selftest_summary(log_text):
    if not log_text or "No self-tests have been logged" in log_text:
        return ("info", "No completed tests found. The test may still be running or logging is unsupported."), None
    lines = [ln for ln in log_text.splitlines() if ln.strip()]
    latest_line = None
    for ln in lines:
        if ln.lstrip().startswith("# ") or re.match(r"^#?\s*\d+\s+", ln):
            latest_line = ln
            break
    if not latest_line:
        for pat, (tag, msg) in SELFTEST_STATUS_PATTERNS:
            if pat.search(log_text):
                return (tag, msg), None
        return ("info", "Could not parse self-test status."), None
    for pat, (tag, msg) in SELFTEST_STATUS_PATTERNS:
        if pat.search(latest_line):
            return (tag, msg), latest_line
    return ("info", "Self-test status present but unrecognized."), latest_line


# -------------------------------
# Main App Class
# -------------------------------
class DiskDigApp(tk.Tk):
    def __init__(self):
        super().__init__()
        try:
            self.call("tk", "appname", "diskdig")
        except Exception:
            pass

        self.title("DiskDig — Drive Diagnostic Tool")
        self.geometry("850x760")
        self.path_map = {}

        self._build_ui()
        self.list_volumes()

    # -------------------------------
    # UI construction (preserved)
    # -------------------------------
    def _build_ui(self):
        frame = ttk.Frame(self, padding=10)
        frame.pack(fill="both", expand=True)

        # Authorization row
        menu_frame = ttk.Frame(frame)
        menu_frame.pack(fill="x", pady=(0, 5))

        auth_frame = ttk.Frame(menu_frame)
        auth_frame.pack(side="left", padx=5)
        ttk.Label(auth_frame, text="Authorization Required:").pack(side="left")
        self.pw_entry = ttk.Entry(auth_frame, width=15, show="*")
        self.pw_entry.pack(side="left", padx=2)
        save_btn = ttk.Button(auth_frame, text="Save", command=self.save_password, style="Auth.TButton")
        save_btn.pack(side="left")

        style = ttk.Style()
        style.configure("Auth.TButton", foreground="white", background="blue")
        style.configure("Action.TButton", background="#444444", foreground="white", font=("TkDefaultFont", 10, "bold"))
        style.map("Action.TButton", background=[("active", "#666666")])

        # Options menu (right)
        menubtn = tk.Menubutton(menu_frame, text="Options", relief="raised")
        menu = tk.Menu(menubtn, tearoff=0)
        menu.add_command(label="Clear Results", command=self.clear_results)
        menu.add_command(label="Refresh Drives", command=self.refresh_all)
        menu.add_separator()
        menu.add_command(label="Exit", command=self.quit)
        menubtn.config(menu=menu)
        menubtn.pack(anchor="ne")

        # Mounted drives
        ttk.Label(frame, text="Select A Currently Mounted Drive", font=("TkDefaultFont", 11, "bold")).pack(anchor="w")
        vol_frame = ttk.Frame(frame)
        vol_frame.pack(fill="x", pady=(0, 5))
        self.volumes_box = tk.Listbox(vol_frame, height=4, font=("TkFixedFont", 11))
        self.volumes_box.pack(side="left", fill="x", expand=True)
        scroll = ttk.Scrollbar(vol_frame, orient="vertical", command=self.volumes_box.yview)
        scroll.pack(side="right", fill="y")
        self.volumes_box.config(yscrollcommand=scroll.set)
        self.volumes_box.bind("<<ListboxSelect>>", self.on_volume_select)

        # SMART Overview
        ttk.Label(frame, text="SMART Overview:", font=("TkDefaultFont", 11, "bold")).pack(anchor="w")
        smart_frame = ttk.Frame(frame)
        smart_frame.pack(fill="both", expand=True, pady=5)
        smart_scroll = tk.Scrollbar(smart_frame, orient="vertical")
        self.smart_box = tk.Text(smart_frame, wrap="word", height=10, yscrollcommand=smart_scroll.set)
        self.smart_box.pack(side="left", fill="both", expand=True)
        smart_scroll.config(command=self.smart_box.yview)
        smart_scroll.pack(side="right", fill="y")

        self.smart_box.tag_configure("warning", background="red", foreground="white")
        self.smart_box.tag_configure("good", background="green", foreground="white")
        self.smart_box.tag_configure("success", background="#d6f5d6", foreground="black")
        self.smart_box.tag_configure("fail", background="#f5d6d6", foreground="black")
        self.smart_box.tag_configure("caution", background="#fff5cc", foreground="black")
        self.smart_box.tag_configure("heading", font=("TkDefaultFont", 11, "bold"))

        def block_typing(event):
            return "break"
        self.smart_box.bind("<Key>", block_typing)

        # Right-click copy menu
        self.smart_menu = tk.Menu(self.smart_box, tearoff=0)
        self.smart_menu.add_command(label="Copy", command=self.copy_smart_selection)
        self.smart_menu.add_command(label="Select All", command=self.select_all_smart)
        self.smart_box.bind("<Button-3>", self.show_smart_menu)
        self.smart_box.bind("<Button-2>", self.show_smart_menu)

        ttk.Separator(frame, orient="horizontal").pack(fill="x", pady=8)

        # Diagnostic tests
        diag_frame = ttk.Frame(frame)
        diag_frame.pack(anchor="w", pady=(0, 10), fill="x")
        ttk.Label(diag_frame, text="Diagnostic Tests", font=("TkDefaultFont", 10, "bold")).pack(side="left", padx=(0, 15))
        ttk.Button(diag_frame, text="Run Short Test", command=lambda: self.start_smart_test("short"), style="Action.TButton").pack(side="left", padx=5)
        ttk.Button(diag_frame, text="Run Long Test", command=lambda: self.start_smart_test("long"), style="Action.TButton").pack(side="left", padx=5)
        ttk.Button(diag_frame, text="View Self-Test Log", command=self.view_selftest_log, style="Action.TButton").pack(side="left", padx=10)
        clear_btn = tk.Button(diag_frame, text="Clear Fields", bg="#b7fcb7", activebackground="#98e898", command=self.clear_results)
        clear_btn.pack(side="left", padx=10)

        # Unmount controls
        btn_row = ttk.Frame(frame)
        btn_row.pack(fill="x", pady=5)
        self.unmount_entry = ttk.Entry(btn_row, width=12)
        self.unmount_entry.insert(0, "/dev/sdX")
        self.unmount_entry.pack(side="right", padx=(5, 0))
        ttk.Button(btn_row, text="Unmount & Stop Drive:", command=self.unmount_drive, style="Action.TButton").pack(side="right", padx=(10, 5))

        # Drive info + output
        ttk.Label(frame, text="Drive Info / Diagnostic Output:", font=("TkDefaultFont", 11, "bold")).pack(anchor="w")
        output_frame = ttk.Frame(frame)
        output_frame.pack(fill="both", pady=5, expand=True)
        yscroll = tk.Scrollbar(output_frame, orient="vertical")
        self.output_box = tk.Text(output_frame, wrap="none", height=8, yscrollcommand=yscroll.set)
        self.output_box.pack(side="left", fill="both", expand=True)
        yscroll.config(command=self.output_box.yview)
        yscroll.pack(side="right", fill="y")

        # Analysis Summary
        ttk.Label(frame, text="Analysis Summary", font=("TkDefaultFont", 10, "bold")).pack(anchor="w")
        self.analysis_box = tk.Text(frame, wrap="word", height=8)
        self.analysis_box.pack(fill="both", expand=False, pady=5)
        self.analysis_box.tag_configure("success", foreground="#156d22")
        self.analysis_box.tag_configure("fail", foreground="#a31212")
        self.analysis_box.tag_configure("caution", foreground="#8a6d00")
        self.analysis_box.tag_configure("info", foreground="#2d4f8a")

    # -------------------------------
    # SMART text copy helpers
    # -------------------------------
    def show_smart_menu(self, event):
        try:
            self.smart_menu.tk_popup(event.x_root, event.y_root)
        finally:
            self.smart_menu.grab_release()

    def copy_smart_selection(self):
        try:
            selection = self.smart_box.get("sel.first", "sel.last")
        except tk.TclError:
            return
        self.clipboard_clear()
        self.clipboard_append(selection)
        self.update()

    def select_all_smart(self):
        self.smart_box.tag_add("sel", "1.0", "end-1c")

    # -------------------------------
    # Authorization
    # -------------------------------
    def save_password(self):
        global sudo_password
        sudo_password = self.pw_entry.get().strip()
        if sudo_password:
            messagebox.showinfo("Authorization", "Password saved for this session.")
        else:
            messagebox.showwarning("Authorization", "No password entered.")

    # -------------------------------
    # Menu actions
    # -------------------------------
    def clear_results(self):
        self.smart_box.delete("1.0", tk.END)
        self.output_box.delete("1.0", tk.END)
        self.analysis_box.delete("1.0", tk.END)

    def refresh_all(self):
        self.clear_results()
        self.list_volumes()

    # -------------------------------
    # Unmount + Power-off
    # -------------------------------
    def unmount_drive(self):
        dev = self.unmount_entry.get().strip()
        if not dev.startswith("/dev/"):
            messagebox.showerror("No Device", "Please enter a valid device path like /dev/sdX")
            return
        try:
            out1 = run_root_task(["udisksctl", "unmount", "-b", dev])
            out2 = run_root_task(["udisksctl", "power-off", "-b", dev])
            combined = (out1 + "\n" + out2).strip()
            messagebox.showinfo("Drive Unmounted", f"Drive {dev} unmounted and powered off.\n\n{combined}")
        except Exception as e:
            messagebox.showerror("Error", f"Failed to unmount/power-off {dev}\n\n{e}")

    # -------------------------------
    # Device resolution helpers
    # -------------------------------
    def resolve_base_device(self, any_path):
        """Return the base device (/dev/sda, /dev/nvme0n1) for a mount or partition."""
        dev_node = any_path
        if not any_path.startswith("/dev/"):
            src = run_cmd(f"findmnt -n -o SOURCE --target '{any_path}'").strip()
            if not src.startswith("/dev/"):
                return None
            dev_node = src
        parent = run_cmd(f"lsblk -no PKNAME {dev_node}").strip()
        if parent:
            candidate = f"/dev/{parent}"
        else:
            candidate = dev_node
        return candidate if os.path.exists(candidate) else None

    # -------------------------------
    # Volume listing + SMART summary
    # -------------------------------
    def list_volumes(self):
        self.volumes_box.delete(0, tk.END)
        self.path_map.clear()
        try:
            df_out = subprocess.check_output(["df", "-hT"], text=True).splitlines()
            for line in df_out[1:]:
                parts = line.split()
                if len(parts) < 7:
                    continue
                device, fstype, size, used, avail, percent, mountpoint = parts[:7]
                if fstype in ("tmpfs", "devtmpfs", "overlay", "cifs", "smbfs"):
                    continue
                if mountpoint.startswith(("/boot", "/proc", "/sys", "/run")):
                    continue
                uuid = run_cmd(f"blkid -s UUID -o value {device}") or "root"
                entry = f"{device:<12} [UUID:{uuid:<12}] {size:>6} total {avail:>6} free"
                self.volumes_box.insert(tk.END, entry)
                self.path_map[entry] = mountpoint
        except Exception as e:
            self.output_box.insert(tk.END, f"Error listing volumes: {e}\n")

    def on_volume_select(self, event):
        sel = self.volumes_box.curselection()
        if not sel:
            return
        entry = self.volumes_box.get(sel[0])
        mountpoint = self.path_map.get(entry)
        if mountpoint:
            self.fetch_drive_info_async(mountpoint)

    def get_drive_info_and_smart(self, path):
        try:
            base = self.resolve_base_device(path)
            if not base:
                return None, "⚠ Could not determine device for selected path.", ""
            lsblk_info = run_cmd(f"lsblk -o NAME,MODEL,SERIAL,VENDOR,SIZE {base}")
            smart_output = run_smartctl_bundle(base)
            details = f"Drive info for {base}:\n\n--- lsblk ---\n{lsblk_info}\n\n--- smartctl (i/A/H) ---\n{smart_output}"
            return base, details, smart_output
        except Exception as e:
            return None, f"⚠ Error getting drive info: {e}", ""

    def fetch_drive_info_async(self, path):
        def worker():
            base, info, smart_output = self.get_drive_info_and_smart(path)
            if base:
                analysis = interpret_smart_summary(smart_output)
                summary_lines = ["SUMMARY OF ANALYSIS"]
                for line in analysis.splitlines():
                    summary_lines.append(line)
                summary_block = "\n".join(summary_lines) + "\n" + "-" * 40 + "\n\n"
            else:
                summary_block = "SUMMARY OF ANALYSIS\n⚠ SMART overview not available.\n\n"
                smart_output = ""

            def update_ui():
                self.smart_box.delete("1.0", tk.END)
                for line in summary_block.splitlines():
                    tag = None
                    if line.strip() == "SUMMARY OF ANALYSIS":
                        tag = "heading"
                    elif "PASSED" in line:
                        tag = "success"
                    elif "FAILED" in line or "Reallocated sectors detected" in line:
                        tag = "fail"
                    elif "Higher failure risk" in line:
                        tag = "caution"
                    if tag:
                        self.smart_box.insert(tk.END, line + "\n", tag)
                    else:
                        self.smart_box.insert(tk.END, line + "\n")
                if smart_output:
                    self.smart_box.insert(tk.END, smart_output + "\n")

                self.output_box.delete("1.0", tk.END)
                self.output_box.insert(tk.END, info + "\n\n")

            self.after(0, update_ui)

        threading.Thread(target=worker, daemon=True).start()

    # -------------------------------
    # SMART self-test functions
    # -------------------------------
    def start_smart_test(self, test_type):
        sel = self.volumes_box.curselection()
        if not sel:
            messagebox.showwarning("No Drive Selected", "Please select a mounted drive first.")
            return
        entry = self.volumes_box.get(sel[0])
        mountpoint = self.path_map.get(entry)
        base = self.resolve_base_device(mountpoint)
        if not base:
            messagebox.showerror("Error", "Could not determine base device for selected mount.")
            return

        self.output_box.delete("1.0", tk.END)
        self.output_box.insert(tk.END, f"Running {test_type} SMART self-test on {base}...\n")

        def worker():
            start_out = run_smart_selftest(base, test_type)
            self.after(0, lambda: self.output_box.insert(tk.END, start_out + "\n"))

            wait_time = 90 if test_type == "short" else 1200
            deadline = time.time() + wait_time
            log = ""

            while time.time() < deadline:
                time.sleep(10)
                log_try = fetch_selftest_log(base)
                if "No self-tests have been logged" not in log_try and "In progress" not in log_try:
                    log = log_try
                    break
            if not log:
                log = fetch_selftest_log(base)

            (tag, msg), latest_line = parse_selftest_summary(log)

            def update_analysis():
                self.analysis_box.insert(tk.END, "\n--- SELF-TEST LOG ---\n")
                self.analysis_box.insert(tk.END, msg + "\n", tag)
                if latest_line:
                    self.analysis_box.insert(tk.END, latest_line + "\n")
                self.analysis_box.insert(tk.END, "\n" + log + "\n")

            self.after(0, update_analysis)

        threading.Thread(target=worker, daemon=True).start()

    def view_selftest_log(self):
        sel = self.volumes_box.curselection()
        if not sel:
            messagebox.showwarning("No Drive Selected", "Please select a mounted drive first.")
            return
        entry = self.volumes_box.get(sel[0])
        mountpoint = self.path_map.get(entry)
        base = self.resolve_base_device(mountpoint)
        if not base:
            messagebox.showerror("Error", "Could not determine base device for selected mount.")
            return
        log = fetch_selftest_log(base)
        self.output_box.delete("1.0", tk.END)
        self.output_box.insert(tk.END, log + "\n")


# -------------------------------
# Main entry
# -------------------------------
if __name__ == "__main__":
    app = DiskDigApp()
    app.mainloop()
