#!/usr/bin/env python3
import os
import re
import sys
import shutil
import tkinter as tk
from tkinter import ttk, filedialog
import subprocess
import threading
import time
from pathlib import Path
from datetime import datetime

try:
    from PIL import Image, ImageTk, ImageOps
    PIL_OK = True
except Exception:
    PIL_OK = False

print("PATH:", os.environ.get('PATH'))

# --- resource_path helper ---
def resource_path(relative_path):
    """Get absolute path to resource, works for dev and PyInstaller."""
    try:
        base_path = sys._MEIPASS  # type: ignore[attr-defined]
    except Exception:
        base_path = os.path.abspath(".")
    return os.path.join(base_path, relative_path)

# Backgrounds inside backgrounds/ folder
LUNAR_PATH   = "/usr/share/yt-dlp/backgrounds/lunar_dunes.gif"
ASTRAL_PATH  = "/usr/share/yt-dlp/backgrounds/astral.gif"
FLOWERS_PATH = "/usr/share/yt-dlp/backgrounds/flowers.gif"

WINDOW_W, WINDOW_H = 1200, 720
YTDLP_CMD = "/usr/local/bin/yt-dlp"   # use system yt-dlp

# Ensure /usr/local/bin is in PATH so standalone can find yt-dlp
if "/usr/local/bin" not in os.environ.get("PATH", "").split(os.pathsep):
    os.environ["PATH"] = os.environ.get("PATH", "") + os.pathsep + "/usr/local/bin"

def safe_text(text: str) -> str:
    """Sanitize text for GUI/log display (not filenames)."""
    return re.sub(r'[^\w\s\.-]', '_', text)

class ThemedApp(tk.Tk):
    def __init__(self):
        super().__init__()

        # 🔑 Force WM_CLASS / app identity early and simply
        try:
            self.call('tk', 'appname', 'youtube-downloader')
        except Exception as e:
            print("appname set failed:", e)

        self.title("Simple Video Downloader for Linux")
        self.geometry(f"{WINDOW_W}x{WINDOW_H}")
        self.resizable(False, False)

        self._bg_photo = None
        self._hover_index = None
        self.bg_label = tk.Label(self)
        self.bg_label.place(x=0, y=0, relwidth=1, relheight=1)

        self.style = ttk.Style(self)
        self.style.theme_use("clam")
        self._palette = {}
        self._setup_styles(light=True)

        # --- URL Label + Entry (separate, no placeholder confusion) ---
        ttk.Label(self, text="Paste URL:", style="App.TLabel").place(x=20, y=24)

        self.url_var = tk.StringVar()
        self.url_entry = ttk.Entry(self, textvariable=self.url_var, style="App.TEntry")
        self.url_entry.place(x=115, y=20, width=WINDOW_W - 135, height=28)
        self._make_context_menu(self.url_entry)

        # --- Fetch Formats button ---
        self.fetch_button = ttk.Button(
            self, text="Fetch Formats", style="Accent.TButton", command=self._fetch_formats
        )
        self.fetch_button.place(x=20, y=70, width=150, height=32)

        # --- Mode tickboxes ---
        self.mode_var = tk.StringVar(value="YouTube")
        self.mode_frame = ttk.Frame(self, style="App.TFrame")
        self.mode_frame.place(x=190, y=70, height=32)

        ttk.Radiobutton(self.mode_frame, text="YouTube", value="YouTube",
                        variable=self.mode_var, style="Segmented.TRadiobutton").pack(side="left", padx=6)
        ttk.Radiobutton(self.mode_frame, text="Other", value="Other",
                        variable=self.mode_var, style="Segmented.TRadiobutton").pack(side="left", padx=6)

        # --- Formats list ---
        self.display_frame = ttk.Frame(self, style="App.TFrame")
        self.display_frame.place(x=20, y=120, width=WINDOW_W - 40, height=120)

        self.format_list = tk.Listbox(
            self.display_frame,
            activestyle="none",
            bg="white",
            fg="black",
            font=("Segoe UI", 10),
            selectbackground="lightgray",
            highlightthickness=0,
            height=6,
            exportselection=False,
            selectmode="browse",
        )
        self.format_list.pack(side="left", fill="both", expand=True)

        scroll1 = ttk.Scrollbar(self.display_frame, orient="vertical", command=self.format_list.yview)
        scroll1.pack(side="right", fill="y")
        self.format_list.configure(yscrollcommand=scroll1.set)

        self.format_list.bind("<Motion>", self._on_hover)
        self.format_list.bind("<Leave>", self._on_leave_list)
        self.format_list.bind("<<ListboxSelect>>", self._on_select)

        # --- Save To section ---
        self.save_var = tk.StringVar(value="Downloads")
        self.custom_dir = None
        self.english_var = tk.BooleanVar(value=False)
        self.playlist_var = tk.BooleanVar(value=False)

        self.save_frame = ttk.Frame(self, style="App.TFrame")
        mid_y = (20 + 28 + 40 + 120 + (WINDOW_H - 220)) // 2
        self.save_frame.place(relx=0.5, y=mid_y, anchor="center")

        ttk.Label(self.save_frame, text="Save To:", style="App.TLabel").grid(row=0, column=0, padx=8)
        for i, opt in enumerate(["Videos", "Downloads", "Desktop"], start=1):
            rb = ttk.Radiobutton(
                self.save_frame,
                text=opt,
                value=opt,
                variable=self.save_var,
                style="Segmented.TRadiobutton",
                command=lambda: self.log(f"Save location set to: {self.get_save_path()}")
            )
            rb.grid(row=0, column=i, padx=8)

        # --- Browse button ---
        browse_btn = ttk.Button(
            self.save_frame,
            text="Browse…",
            style="Accent.TButton",
            command=self.browse_folder
        )
        browse_btn.grid(row=0, column=4, padx=8)

        ttk.Label(self.save_frame, text="|", style="App.TLabel").grid(row=0, column=5, padx=8)

        english_cb = ttk.Checkbutton(
            self.save_frame,
            text="English If Available",
            variable=self.english_var,
            style="App.TCheckbutton"
        )
        english_cb.grid(row=0, column=6, padx=8)

        playlist_cb = ttk.Checkbutton(
            self.save_frame,
            text="Allow Playlists",
            variable=self.playlist_var,
            style="App.TCheckbutton"
        )
        playlist_cb.grid(row=0, column=7, padx=8)

        # --- Download + Reset buttons ---
        self.download_button = ttk.Button(
            self, text="Download Selected", style="Accent.TButton", command=self._download_selected
        )
        self.download_button.place(x=20, y=WINDOW_H - 260, width=180, height=32)
        self.download_button.state(["disabled"])

        self.reset_button = ttk.Button(
            self, text="Reset", style="Accent.TButton", command=self._reset_all
        )
        self.reset_button.place(x=210, y=WINDOW_H - 260, width=120, height=32)

        # --- Update yt-dlp button ---
        self.update_button = ttk.Button(
            self, text="Update yt-dlp", style="Update.TButton", command=self._update_ytdlp
        )
        self.update_button.place(x=340, y=WINDOW_H - 260, width=160, height=32)

        # --- Output log ---
        self.output_frame = ttk.Frame(self, style="App.TFrame")
        self.output_frame.place(x=20, y=WINDOW_H - 220, width=WINDOW_W - 40, height=120)

        self.output_display = tk.Text(
            self.output_frame,
            wrap="word",
            height=6,
            bg="white",
            fg="black",
            font=("Segoe UI", 10),
        )
        self.output_display.pack(side="left", fill="both", expand=True)

        scroll2 = ttk.Scrollbar(self.output_frame, orient="vertical", command=self.output_display.yview)
        scroll2.pack(side="right", fill="y")
        self.output_display.configure(yscrollcommand=scroll2.set)

        # --- Theme picker / spinner wrapper ---
        self.theme_var = tk.StringVar(value="Light")
        self.theme_frame = ttk.Frame(self, style="App.TFrame")
        self.theme_frame.place(relx=0.5, rely=1.0, anchor="s", width=700, height=80)

        self._build_theme_selector()

        self.apply_theme("Light")
        self.log("Application started")

    # ---------- Custom Warning Popup ----------
    def _show_warning(self, title: str, message: str):
        bg = self._palette.get("bg", "#eaeaea")

        popup = tk.Toplevel(self)
        popup.title(title)
        popup.resizable(False, False)
        popup.configure(bg=bg)

        w, h = 420, 200
        x = self.winfo_x() + (self.winfo_width() // 2) - (w // 2)
        y = self.winfo_y() + (self.winfo_height() // 2) - (h // 2)
        popup.geometry(f"{w}x{h}+{x}+{y}")

        container = ttk.Frame(popup, style="App.TFrame")
        container.pack(fill="both", expand=True, padx=20, pady=16)

        ttk.Label(
            container,
            text=title,
            style="App.TLabel",
            font=("Segoe UI", 14, "bold"),
            anchor="center",
            justify="center"
        ).pack(fill="x", pady=(0, 10))

        ttk.Label(
            container,
            text=message,
            style="App.TLabel",
            wraplength=w-60,
            anchor="center",
            justify="center"
        ).pack(fill="x", pady=(0, 20))

        ttk.Button(container, text="OK", style="Accent.TButton", command=popup.destroy)\
            .pack(pady=(6, 0))

        popup.transient(self)
        popup.grab_set()
        popup.focus_force()
        popup.bind("<Return>", lambda _e: popup.destroy())
        popup.bind("<Escape>", lambda _e: popup.destroy())

        self.wait_window(popup)

    # ---------- Browse folder ----------
    def browse_folder(self):
        folder = filedialog.askdirectory(parent=self, title="Select Download Folder")
        if folder:
            self.custom_dir = Path(folder)
            self.save_var.set("Custom")
            self.log(f"Save location set to custom: {self.custom_dir}")

    # ---------- Spinner handling ----------
    def _replace_with_spinner(self, text="Downloading…"):
        for w in self.theme_frame.winfo_children():
            w.destroy()
        spinner_frame = ttk.Frame(self.theme_frame, style="App.TFrame")
        spinner_frame.pack(expand=True)
        lbl = ttk.Label(spinner_frame, text=text, style="App.TLabel")
        lbl.pack(pady=5)
        pb = ttk.Progressbar(spinner_frame, mode="indeterminate", length=200)
        pb.pack(pady=5)
        pb.start(10)
        self._spinner_label = lbl
        self._spinner_pb = pb

    def _restore_theme_selector(self):
        for w in self.theme_frame.winfo_children():
            w.destroy()
        self._build_theme_selector()

    def _build_theme_selector(self):
        ttk.Label(self.theme_frame, text="Theme", style="App.TLabel").pack(pady=(0, 5))
        self._seg_container = ttk.Frame(self.theme_frame, style="App.TFrame")
        self._seg_container.pack(fill="x")
        for i, (label, value) in enumerate(
            [("Light", "Light"), ("Dark", "Dark"), ("Flowers", "Flowers"),
             ("Lunar Dunes", "Lunar"), ("Astral", "Astral")]
        ):
            rb = ttk.Radiobutton(
                self._seg_container,
                text=label,
                value=value,
                variable=self.theme_var,
                style="Segmented.TRadiobutton",
                command=self.apply_theme_from_var
            )
            rb.grid(row=0, column=i, sticky="nsew")
            self._seg_container.grid_columnconfigure(i, weight=1)

    # ---------- Styles ----------
    def _setup_styles(self, light: bool, dark=False):
        if light:
            bg, fg, entry_bg, accent, hover = "#eaeaea", "#1a1a1a", "#f2f2f2", "#3b82f6", "#dfe8ff"
        elif dark:
            bg, fg, entry_bg, accent, hover = "#1e1e1e", "#e8e8e8", "#2a2a2a", "#60a5fa", "#30343c"
        else:
            bg, fg, entry_bg, accent, hover = "#0f1115", "#e8e8e8", "#1a1d24", "#60a5fa", "#262a33"

        self._palette = {"bg": bg, "fg": fg, "entry_bg": entry_bg, "accent": accent, "hover": hover}

        self.style.configure("App.TFrame", background=bg)
        self.style.configure("App.TLabel", background=bg, foreground=fg, font=("Segoe UI", 11))
        self.style.configure("App.TEntry", fieldbackground=entry_bg, foreground=fg, insertcolor=fg)
        self.style.configure("App.TCheckbutton", background=bg, foreground=fg, font=("Segoe UI", 10))
        self.style.configure("Accent.TButton", background=accent, foreground="white",
                             borderwidth=0, focusthickness=0, font=("Segoe UI", 10))
        self.style.map("Accent.TButton", background=[("active", hover), ("pressed", accent)])

        self.style.configure("Update.TButton",
                             background="#77dd77",
                             foreground="black",
                             borderwidth=0,
                             focusthickness=0,
                             font=("Segoe UI", 10))
        self.style.map("Update.TButton",
                       background=[("active", "#66cc66"), ("pressed", "#55aa55")])

        self.style.configure("Segmented.TRadiobutton",
                             background=bg,
                             foreground=fg,
                             font=("Segoe UI", 10),
                             padding=(10, 4),
                             relief="flat")
        self.style.map("Segmented.TRadiobutton",
                       background=[("selected", bg), ("!selected", bg)],
                       relief=[("pressed", "flat"), ("!pressed", "flat")])

    # ---------- Logging ----------
    def log(self, text):
        timestamp = datetime.now().strftime("%H:%M:%S")
        safe_line = safe_text(text)
        self.output_display.insert("end", f"[{timestamp}] {safe_line}\n")
        self.output_display.see("end")

    # ---------- Reset ----------
    def _reset_all(self):
        self.url_var.set("")
        self.format_list.delete(0, "end")
        self.output_display.delete("1.0", "end")
        self.download_button.state(["disabled"])
        self.english_var.set(False)
        self.mode_var.set("YouTube")
        # ⬇️ Do NOT reset self.custom_dir or self.save_var
        self.log("Fields cleared (download directory preserved).")

    # ---------- Context Menu ----------
    def _make_context_menu(self, widget):
        menu = tk.Menu(self, tearoff=0)
        for lbl, ev in [("Cut", "<<Cut>>"), ("Copy", "<<Copy>>"), ("Paste", "<<Paste>>")]:
            menu.add_command(label=lbl, command=lambda e=ev: widget.event_generate(e))
        widget.bind("<Button-3>", lambda e: menu.tk_popup(e.x_root, e.y_root))

    # ---------- Theme ----------
    def apply_theme_from_var(self):
        self.apply_theme(self.theme_var.get())

    def apply_theme(self, name: str):
        n = name.lower()
        if n == "light":
            self._setup_styles(light=True); self._set_solid("#eaeaea")
        elif n == "dark":
            self._setup_styles(light=False, dark=True); self._set_solid("#1e1e1e")
        elif n == "flowers":
            self._setup_styles(light=False, dark=False); self._set_image(FLOWERS_PATH)
        elif n == "lunar":
            self._setup_styles(light=False, dark=False); self._set_image(LUNAR_PATH)
        elif n == "astral":
            self._setup_styles(light=False, dark=False); self._set_image(ASTRAL_PATH)
        else:
            self._setup_styles(light=True); self._set_solid("#eaeaea")

    def _set_solid(self, color: str):
        self._bg_photo = None
        self.bg_label.configure(bg=color, image="")

    def _set_image(self, path: str):
        if not os.path.exists(path):
            return self._set_solid("#0f1115")
        try:
            if PIL_OK:
                img = Image.open(path)
                fitted = ImageOps.fit(img, (WINDOW_W, WINDOW_H), method=Image.LANCZOS)
                self._bg_photo = ImageTk.PhotoImage(fitted)
            else:
                self._bg_photo = tk.PhotoImage(file=path)
        except Exception:
            return self._set_solid("#0f1115")
        self.bg_label.configure(image=self._bg_photo)

    # ---------- URL Cleaning ----------
    def clean_url(self, url: str) -> str:
        core = url.split("?")[0].split("#")[0]
        if "rumble.com" in core and "/video/" in core:
            parts = core.split("/video/")
            if len(parts) > 1:
                vid_id = parts[1].split("/")[0]
                return f"https://rumble.com/video/{vid_id}"
        return core

    # ---------- Fetch Formats ----------
    def _fetch_formats(self):
        url = self.url_var.get().strip()
        if not url:
            return self._show_warning("Missing URL", "Please paste a URL first.")
        if self.mode_var.get() == "Other":
            url = self.clean_url(url)

        if ("?list=" in url or "&list=" in url) and not self.playlist_var.get():
            return self._show_warning(
                "Playlist Detected!",
                "Playlists often contain unlisted files and may be 100+ files - tick 'Allow Playlists' to approve."
            )

        self.fetch_button.state(["disabled"])
        self.fetch_button.config(text="Fetching…")
        self.log(f"Fetching formats for: {url}")
        threading.Thread(target=self._run_fetch_formats, args=(url,), daemon=True).start()

    def _run_fetch_formats(self, url):
        lines = []
        try:
            process = subprocess.Popen(
                [YTDLP_CMD, "--no-warnings", "--list-formats", "--playlist-items", "1", url],
                stdout=subprocess.PIPE, stderr=subprocess.STDOUT, text=True, bufsize=1
            )
            for line in process.stdout:
                if line.strip():
                    self.log(line.strip())
                    lines.append(line.rstrip("\n"))
            process.wait()
        except Exception as e:
            self.log(f"Error fetching formats: {e}")

        self.after(0, lambda: self._populate_formats(lines))

    def _populate_formats(self, lines):
        self.format_list.delete(0, "end")
        for line in lines:
            self.format_list.insert("end", line)
        self.download_button.state(["disabled"])
        self.fetch_button.config(text="Fetch Formats")
        self.fetch_button.state(["!disabled"])

    # ---------- Download ----------
    def _download_selected(self):
        url = self.url_var.get().strip()
        if not url:
            return self._show_warning("Missing URL", "Please paste a URL first.")

        if self.mode_var.get() == "Other":
            url = self.clean_url(url)

        if ("?list=" in url or "&list=" in url) and not self.playlist_var.get():
            return self._show_warning(
                "Playlist Detected!",
                "Playlists often contain unlisted files and may be 100+ files"
            )

        selection = self.format_list.curselection()
        if not selection:
            return self._show_warning("No Format Selected", "Please select a format to download.")

        # ---- NEW LOGIC: match CLI behavior ----
        line = self.format_list.get(selection[0])
        fmt_code = line.split()[0]
        lower = line.lower()

        is_audio_only = ("audio only" in lower) or ("audio-only" in lower)
        is_video_only = ("video only" in lower) or ("video-only" in lower)

        if is_video_only:
            if self.english_var.get():
                audio_pref = (
                    "bestaudio[language=en][ext=m4a]/"
                    "bestaudio[language=en][acodec!=?opus]/"
                    "bestaudio[ext=m4a]/"
                    "bestaudio[acodec!=?opus]/"
                    "bestaudio"
                )
            else:
                audio_pref = (
                    "bestaudio[ext=m4a]/"
                    "bestaudio[acodec!=?opus]/"
                    "bestaudio"
                )
            fmt_string = f"{fmt_code}+{audio_pref}"
        else:
            fmt_string = fmt_code

        save_dir = self.get_save_path()
        save_dir.mkdir(parents=True, exist_ok=True)
        out_tpl = os.path.join(str(save_dir), "%(title)s.%(ext)s")

        self._replace_with_spinner("Downloading…")
        self.log(f"Downloading with format '{fmt_string}' to {save_dir}")
        threading.Thread(target=self._run_download, args=(fmt_string, out_tpl, url), daemon=True).start()

    def _run_download(self, fmt, out_tpl, url):
        try:
            process = subprocess.Popen(
                [YTDLP_CMD, "--newline", "--no-color", "-f", fmt, "-o", out_tpl, url],
                stdout=subprocess.PIPE, stderr=subprocess.STDOUT, text=True, bufsize=1
            )
            for line in process.stdout:
                if line.strip():
                    self.log(line.strip())
            process.wait()

            if process.returncode == 0:
                self._spinner_label.config(text="Download Complete ✅")
            else:
                self._spinner_label.config(text=f"Download failed (exit {process.returncode})")
            self._spinner_pb.stop()
            time.sleep(1.2)

        except Exception as e:
            self.log(f"Download error: {e}")
        finally:
            self._restore_theme_selector()

    # ---------- Update yt-dlp ----------
    def _update_ytdlp(self):
        if not shutil.which("pkexec"):
            self.log("pkexec not found. Install policykit-1 or run this app with privileges.")
            return
        self.log("Starting yt-dlp update…")
        threading.Thread(target=self._run_update, daemon=True).start()

    def _run_update(self):
        script_lines = [
            "set -e",
            "echo [root] Relaxing perms on /usr/local/bin to 777...",
            "chmod -R 777 /usr/local/bin",
            "echo [root] Running yt-dlp -U...",
            "/usr/local/bin/yt-dlp -U",
            "echo [root] Restoring perms on /usr/local/bin to 755...",
            "chmod -R 755 /usr/local/bin",
            "echo [root] Update finished."
        ]
        script = "\n".join(script_lines)
        try:
            process = subprocess.Popen(
                ["pkexec", "bash", "-c", script],
                stdout=subprocess.PIPE, stderr=subprocess.STDOUT, text=True, bufsize=1
            )
            for line in process.stdout:
                if line:
                    self.log(line.rstrip("\n"))
            process.wait()
            if process.returncode == 0:
                self.log("yt-dlp update complete ✅")
            else:
                self.log(f"yt-dlp update failed (exit {process.returncode})")
        except Exception as e:
            self.log(f"Update error: {e}")

    # ---------- Hover for formats list ----------
    def _on_hover(self, e):
        if self.format_list.size() == 0:
            return
        try:
            idx = self.format_list.nearest(e.y)
        except Exception:
            return
        if idx < 0:
            return
        if idx == self._hover_index:
            return
        if self._hover_index is not None and self._hover_index not in self.format_list.curselection():
            try:
                self.format_list.itemconfig(self._hover_index, bg="white")
            except Exception:
                pass
        if idx not in self.format_list.curselection():
            try:
                self.format_list.itemconfig(idx, bg="#e6e6e6")
            except Exception:
                pass
        self._hover_index = idx

    def _on_leave_list(self, _e):
        if self._hover_index is not None and self._hover_index not in self.format_list.curselection():
            try:
                self.format_list.itemconfig(self._hover_index, bg="white")
            except Exception:
                pass
        self._hover_index = None

    def _on_select(self, _e):
        if self.format_list.curselection():
            self.download_button.state(["!disabled"])
        else:
            self.download_button.state(["disabled"])

    # ---------- Path Resolution ----------
    def get_save_path(self):
        if self.save_var.get() == "Custom" and self.custom_dir:
            return self.custom_dir
        home = Path.home()
        return {
            "Videos": home / "Videos",
            "Desktop": home / "Desktop"
        }.get(self.save_var.get(), home / "Downloads")


if __name__ == "__main__":
    ThemedApp().mainloop()
